// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001, 2002

// this file contains the (supposedly) Win32-independent
// core simulator.  It is responsible for the non-gui
// portions of the emulation.

#include <stdio.h>
#include <string.h>

#include "solace.h"
#include "solace_intf.h"
#include "script.h"	// script processing routines
#include "wintape.h"	// interface to virtual tape drives
#include "vdisk_svn.h"	// interface to virtual northstar disk system


#define BRKPT_ADDR(addr) (brkpt_map[(addr)>>5] & (1 << ((addr) & 31)))
#define BRKPT_IO(port)   (brkpt_iomap[(port)])

#define SCRIPT_KB_SUPPTIMER 100	// ms to wait for idleness between script commands


// ------------------- forward declarations ----------------------

static void WrZ80_norm(word Addr, byte Value);
static byte RdZ80_norm(word Addr);
static byte InZ80_norm(word Port);
static void OutZ80_norm(word Port, byte Value);

write_sub_t WrZ80_ROM;
write_sub_t WrZ80_display;

static word Exec8080_brkpt(Z80 *R);
static void WrZ80_brkpt(word Addr, byte Value);
static byte RdZ80_brkpt(word Addr);
//static byte OpZ80_brkpt(word Addr);
static byte InZ80_brkpt(word Port);
static void OutZ80_brkpt(word Port, byte Value);

void UpdateTimerInfo(void);
static void handle_scanadv(uint32 arg1, uint32 arg2);
static void handle_kb_stat(uint32 arg1, uint32 arg2);
static void handle_svt_rdtimer(uint32 arg1, uint32 arg2);
static void handle_svt_wrtimer(uint32 arg1, uint32 arg2);
static void handle_kb_supp(uint32 arg1, uint32 arg2);
static void report_curstate(void);

static byte handle_IO_In_F8(word Addr);
static byte handle_IO_In_F9(word Addr);
static byte handle_IO_In_FA(word Addr);
static byte handle_IO_In_FB(word Addr);
static byte handle_IO_In_FC(word Addr);
static byte handle_IO_In_FD(word Addr);
static byte handle_IO_In_FE(word Addr);
static byte handle_IO_In_FF(word Addr);

static void handle_IO_Out_F8(word Addr, byte Value);
static void handle_IO_Out_F9(word Addr, byte Value);
static void handle_IO_Out_FA(word Addr, byte Value);
static void handle_IO_Out_FB(word Addr, byte Value);
static void handle_IO_Out_FD(word Addr, byte Value);
static void handle_IO_Out_FE(word Addr, byte Value);
static void handle_IO_Out_FF(word Addr, byte Value);

// ------------------- state ----------------------

// motherboard state
sysstate_t sysstate;

// simulation state
simstate_t simstate;

// memimage holds the 64 KB memory contents.
// in order to flexibly allow dynamically mapping memory-mapped
// devices to the emulator, the memory space is broken up into
// 256 "pages", with separate handlers for reads and writes.
// if the handler is NULL, we simply read or write memimage.
read_sub_t  *memmapRd[256];
write_sub_t *memmapWr[256];
byte memimage[65536];

// the emulator allows dynamically mapped I/O devices.
read_sub_t  *memmapIn[256];
write_sub_t *memmapOut[256];

// z80 state
Z80 Z80Regs;


// ---------------------------------------
// Abstract the 8080 register set so that interface
// doesn't need to expose z80.h

// read a register or register pair
int
CPU_RegRead(cpureg_t reg)
{
    int rv = 0;
    switch (reg) {
	case CPU_REG_A:  rv = Z80Regs.AF.B.h; break;
	case CPU_REG_F:  rv = Z80Regs.AF.B.l; break;
	case CPU_REG_B:  rv = Z80Regs.BC.B.h; break;
	case CPU_REG_C:  rv = Z80Regs.BC.B.l; break;
	case CPU_REG_D:  rv = Z80Regs.DE.B.h; break;
	case CPU_REG_E:  rv = Z80Regs.DE.B.l; break;
	case CPU_REG_H:  rv = Z80Regs.HL.B.h; break;
	case CPU_REG_L:  rv = Z80Regs.HL.B.l; break;
	case CPU_REG_AF: rv = Z80Regs.AF.W;   break;
	case CPU_REG_BC: rv = Z80Regs.BC.W;   break;
	case CPU_REG_DE: rv = Z80Regs.DE.W;   break;
	case CPU_REG_HL: rv = Z80Regs.HL.W;   break;
	case CPU_REG_PC: rv = Z80Regs.PC.W;   break;
	case CPU_REG_SP: rv = Z80Regs.SP.W;   break;
	default: ASSERT(0);
    }
    return rv;
}

// trigger an interrupt, and specify a reason
void
CPU_Break(brkpt_t cause)
{
    Z80Regs.brkpt  = 1;		// set breakpoint flag
    Z80Regs.reason = cause;	// reason why we're stopping
}


// ---------------------------------------
// Logging routines for debugging

#if 0
static char logfilename[256];

static void
OpenLog(char *filename)
{
    FILE *fp;
    strcpy(logfilename, filename);
    fp = fopen(filename, "w");
    fclose(fp);
}

static void
Log(char *str) {
    FILE *fp;
    fp = fopen(logfilename, "a");
    fprintf(fp, str);
    fclose(fp);
}
#endif


// ---------------------------------------

void
MapRdRange(int low_addr, int high_addr, read_sub_t fcn)
{
    int page_low  = (low_addr  >> 8);
    int page_high = (high_addr >> 8);
    int i;

    ASSERT(page_low  >= 0 && page_low  < 256);
    ASSERT(page_high >= 0 && page_high < 256);

    for(i=page_low; i<= page_high; i++)
	memmapRd[i] = fcn;
}

void
MapWrRange(int low_addr, int high_addr, write_sub_t fcn)
{
    int page_low  = (low_addr  >> 8);
    int page_high = (high_addr >> 8);
    int i;

    ASSERT(page_low  >= 0 && page_low  < 256);
    ASSERT(page_high >= 0 && page_high < 256);

    for(i=page_low; i<= page_high; i++)
	memmapWr[i] = fcn;
}


void
MapInRange(int low_addr, int high_addr, read_sub_t fcn)
{
    int i;

    ASSERT(low_addr  >= 0 && low_addr  <= 0xFF);
    ASSERT(high_addr >= 0 && high_addr <= 0xFF);

    for(i=low_addr; i <= high_addr; i++)
	memmapIn[i] = fcn;
}

void
MapOutRange(int low_addr, int high_addr, write_sub_t fcn)
{
    int i;

    ASSERT(low_addr  >= 0 && low_addr  <= 0xFF);
    ASSERT(high_addr >= 0 && high_addr <= 0xFF);

    for(i=low_addr; i <= high_addr; i++)
	memmapOut[i] = fcn;
}

// initialize state that isn't affected by Cold/Warm reset
// it is only ever called once.
void
Sys_Init(void)
{
    int i;

    // set CPU speed: 2.04 MHz by default
    Sys_Set8080Speed(MHZ_2_04);

    // dipswitch settings
    Sys_SetDipSwitch(1, SW1_POLARITY | SW1_SOLIDCUR);	// display properties
    Sys_SetDipSwitch(2, 0x00);				// sense switches
    Sys_SetDipSwitch(3, 0x20);				// serial port baud rate: 1200 baud default
    Sys_SetDipSwitch(4, 0x28);				// serial port control: 8b, no parity, 1 stop default

    Sys_ThrottleScheduler(Throttle_NORMAL);

    // pointers to memory read and write
    // later on we may switch these for debugging
    simstate.exec8080_func = Exec8080;
    simstate.diz80_func = RdZ80_norm;
    simstate.opz80_func = RdZ80_norm;
    simstate.rdz80_func = RdZ80_norm;
    simstate.wrz80_func = WrZ80_norm;

    // initialize the memory mapped handlers to defaults
    MapRdRange(0x0000, 0xFFFF, NULL);
    MapWrRange(0x0000, 0xFFFF, NULL);

    // set up special ranges for memory mappers
    MapWrRange(0xC000, 0xC7FF, WrZ80_ROM);
    MapWrRange(0xCC00, 0xCFFF, WrZ80_display);

    // initialize the I/O handlers to defaults
    MapInRange (0x00, 0xFF, NULL);
    MapOutRange(0x00, 0xFF, NULL);

    // set up ranges for motherboard devices
    MapInRange(0xF8, 0xF8, handle_IO_In_F8);
    MapInRange(0xF9, 0xF9, handle_IO_In_F9);
    MapInRange(0xFA, 0xFA, handle_IO_In_FA);
    MapInRange(0xFB, 0xFB, handle_IO_In_FB);
    MapInRange(0xFC, 0xFC, handle_IO_In_FC);
    MapInRange(0xFD, 0xFD, handle_IO_In_FD);
    MapInRange(0xFE, 0xFE, handle_IO_In_FE);
    MapInRange(0xFF, 0xFF, handle_IO_In_FF);

    MapOutRange(0xF8, 0xF8, handle_IO_Out_F8);
    MapOutRange(0xF9, 0xF9, handle_IO_Out_F9);
    MapOutRange(0xFA, 0xFA, handle_IO_Out_FA);
    MapOutRange(0xFB, 0xFB, handle_IO_Out_FB);
    // out 0xFC is not used on motherboard
    MapOutRange(0xFD, 0xFD, handle_IO_Out_FD);
    MapOutRange(0xFE, 0xFE, handle_IO_Out_FE);
    MapOutRange(0xFF, 0xFF, handle_IO_Out_FF);

    // init script support
    script_initpkg();
    simstate.kb_script    =  0;
    simstate.kb_sh        = -1;
    simstate.kb_suppress  =  0;
    simstate.kb_supptimer = -1;

    // reset simulation scheduler
    TimerInit();
    sysstate.scrolltimer   =  0;
    simstate.hscrolltimer  = -1;
    sysstate.scanadv       =  0;
    simstate.hscanadv      = -1;
    simstate.hkb_stat      = -1;
    simstate.vt_rdtimer[0] = -1;
    simstate.vt_rdtimer[1] = -1;
    simstate.vt_wrtimer[0] = -1;
    simstate.vt_wrtimer[1] = -1;

    // although the Sol is affected by reset (and is handled in Sys_Reset(),
    // the tape player state isn't affected by reset.
    simstate.vt_realtime = 1;
    for(i=0; i<2; i++) {
	sysstate.vt[i].playstate   = EMPTY;
	sysstate.vt[i].tape        = NULL;
	sysstate.vt[i].motor_on    = 0;
	sysstate.vt[i].motor_force = 0;
    }

    // virtual disk emulation
    svd_emu_init();
#if 0
    // install the card in the system
    i = svd_install_driver(0xE800);
    if (i != SVD_OK)
	UI_Alert("Error installing NS disk controller");
#endif

    // audio subsystem
    intaudio_init();

    // remove all breakpoints
    breakpoint_init();

    // turn on fan noise
    Sys_AudioEvent(SFXE_FAN_ON);
}


// generate audio samples this fast
void
Sys_SetAudioSamplerate(int rate)
{
    intaudio_samplerate(rate);
}

// slow down emulated 8080 (keeps synchronization with audio rate)
void
Sys_ThrottleScheduler(int throttle)
{
    simstate.throttle = throttle;
}


// this routine sets up breakpoint-related crap just before
// resuming execution.
static void
startrun(void)
{
    // generate a bitmap of any active breakpoints
    breakpoint_genmap();

    // we select the fast function if there is no breakpoint
    // of that type, otherwise we pick the slow one that tests
    // for breakpoints on each access.

    if ((simstate.runstate == STEP) || // in case we set tmp breakpoint
	any_breakpoints(BRK_TEMP) ||
	any_breakpoints(BRK_OP) ||
        any_breakpoints(BRK_RD) ||
        any_breakpoints(BRK_WR) ||
        any_breakpoints(BRK_RD16) ||
        any_breakpoints(BRK_WR16) ||
        any_breakpoints(BRK_IN) ||
        any_breakpoints(BRK_OUT)
       )
	simstate.exec8080_func = Exec8080_brkpt;
    else
	simstate.exec8080_func = Exec8080;

    if (any_breakpoints(BRK_OP) || any_breakpoints(BRK_TEMP))
	simstate.opz80_func = RdZ80_norm;	// OpZ80_brkpt; not used
    else
	simstate.opz80_func = RdZ80_norm;

    if (any_breakpoints(BRK_RD) || any_breakpoints(BRK_RD16))
	simstate.rdz80_func = RdZ80_brkpt;
    else
	simstate.rdz80_func = RdZ80_norm;

    if (any_breakpoints(BRK_WR) || any_breakpoints(BRK_WR16))
	simstate.wrz80_func = WrZ80_brkpt;
    else
	simstate.wrz80_func = WrZ80_norm;

    if (any_breakpoints(BRK_IN))
	simstate.inz80_func = InZ80_brkpt;
    else
	simstate.inz80_func = InZ80_norm;

    if (any_breakpoints(BRK_OUT))
	simstate.outz80_func = OutZ80_brkpt;
    else
	simstate.outz80_func = OutZ80_norm;

    // reset debugging state
    Z80Regs.brkpt = 0;
}


void
bkptchange_coresim(void)
{
    startrun();
}


// simulate cold/warm reset
void
Sys_Reset(int warm_reset)
{
    char *msg = "If you see this message, the ROM didn't load";
    unsigned int a;
    int i;

    // init board state
    sysstate.scrollbase  = 0;
    sysstate.scrollshade = 0;
    sysstate.kb_stat     = 0;
    sysstate.kb_key      = 0x00;

    // reset cancels any pending script
    script_closepkg();
    simstate.kb_script    =  0;
    simstate.kb_sh        = -1;
    simstate.kb_suppress  =  0;
    simstate.kb_supptimer = -1;

    if (!warm_reset) {
	// init memory image
	for(a=0; a<65536; a++) {
	    memimage[a] = 0x76;	// HALT instruction
	}
#if 1
	for(a=0xCC00; a<0xCC00+strlen(msg); a++) {
	    memimage[a] = msg[a-0xCC00];
	}
#else
	// initialize with the character set.  useful for debugging
	// the font, although to be most useful, inhibit subsequent
	// ROM load, which will wipe the video display memory clean.
	{
	    int r, c;
	    for(a=0xCC00; a<0xCC00+1024; a++) { memimage[a] = 0x20; }
	    for(r=0; r<16; r++) {
		for(c=0; c<16; c++) {
		    memimage[0xCC00 + r*64+3*c+1] = 16*r+c;
		}
	    }
	}
#endif
    }

    // most virtual tape state is unaffected by reset,
    // as it is off-board
    sysstate.vt_TTBE    = 1;	// tape transmit buffer empty
    sysstate.vt_rxready = 0;	// tape UART state
    sysstate.vt_overrun = 0;	// tape UART state
    sysstate.vt_framing = 0;	// tape UART state
    sysstate.vt_baud    = 1200;	// tape UART control
    for(i=0; i<2; i++)
	sysstate.vt[i].motor_on = 0;	// motor relay

    // reset the ns disk subsystem
    svd_board_reset();

#if 0
    { // for debugging
	int stat;
	stat = Sys_TapeFile(1, VTF_LoadTape, "..\\binaries\\foo.svt");
	printf("TapeFile stat=%d\n", stat);
	//stat = vtape_destroy(vtape);
	//printf("vtape destroy stat=%d\n", stat);
    }
#endif

    // reset the processor state
    ResetZ80(&Z80Regs);
    // the Sol memory decoder forces the first four memory
    // ops to read from 0xC0xx instead of 0x00xx.  Rather
    // than complicating the fetcher, we just pretend reset
    // occurs at 0xC000, the address of the personality module.
    Z80Regs.PC.W = 0xC000;

    if (UI_DbgWinActive()) {
	if (simstate.runstate != RUNNING)
	    UI_DbgWin(DW_Update, UPDATE_ALL);
    } else {
	simstate.runstate = RESET;
    }

    simstate.prog_detect = 0;
}


void
Sys_LoadDefaultROM(void)
{
    int i;
    for(i=0; i<0x800; i++)
	memimage[0xC000+i] = solos_rom[i];
}


// ------------- set/get various states --------------
// abstract our data structs from user interface code

void
Sys_SetDipSwitch(int swtch, byte value)
{
    switch (swtch) {
	case 1: sysstate.dipsw1 = (sw1_t)value; break;
	case 2: sysstate.dipsw2 = value; break;
	case 3: sysstate.dipsw3 = value; break;
	case 4: sysstate.dipsw4 = value; break;
	default:
	    printf("Error: attempt to set dipswitch %d\n", swtch);
    }
}


byte
Sys_GetDipswitch(int swtch)
{
    switch (swtch) {
	case 1: return sysstate.dipsw1;
	case 2: return sysstate.dipsw2;
	case 3: return sysstate.dipsw3;
	case 4: return sysstate.dipsw4;
	default:
	    printf("Error: attempt to get dipswitch %d\n", swtch);
	    return 0x00;
    }
}


void
Sys_Set8080Speed(speed_t clkrate)
{
    simstate.up_speed = clkrate;
    UpdateTimerInfo();
}

speed_t
Sys_Get8080Speed(void)
{
    return simstate.up_speed;
}


// get play state of indicated unit
void
Sys_GetTapeProp(int unit, int prop, int *param)
{
    ASSERT(unit == 0 || unit == 1);
    switch (prop) {
	case TPROP_REALTIME:
	    *param = simstate.vt_realtime;	// ignores unit
	    break;
	case TPROP_PLAYSTATE:
	    *param = sysstate.vt[unit].playstate;
	    break;
	case TPROP_FORCEMOTOR:
	    *param = sysstate.vt[unit].motor_force;
	    break;
	default:
	    (void)vtape_getprop(sysstate.vt[unit].tape, prop, param);
	    break;
    }
}

// set play state of indicated unit
void
Sys_SetTapeProp(int unit, int prop, int param)
{
    ASSERT(unit == 0 || unit == 1);
    switch (prop) {
	case TPROP_REALTIME:
	    simstate.vt_realtime = param;	// ignores unit
	    break;
	case TPROP_PLAYSTATE:
	    sysstate.vt[unit].playstate = param;
	    WinTapeNotify(unit, WT_PLAYSTATE, 0);
	    break;
	case TPROP_FORCEMOTOR:
	    sysstate.vt[unit].motor_force = param;
	    break;
	default:
	    (void)vtape_setprop(sysstate.vt[unit].tape, prop, param);
	    break;
    }
}


// transfer virtual tapes between disk and memory.
// this function does as it is told and does no double
// checking to see if there is a dirty tape or whatever
// already in there.
// returns 0 if OK, !=0 on error.
int
Sys_TapeFile(int unit, int action, char *filename)
{
    int stat, pos;

    ASSERT(unit == 0 || unit == 1);

    if (simstate.vt_rdtimer[unit] >= 0)
	TimerKill(simstate.vt_rdtimer[unit]);
    if (simstate.vt_wrtimer[unit] >= 0)
	TimerKill(simstate.vt_wrtimer[unit]);

    switch (action) {

	case VTF_NewTape:
	    // get rid of old tape if it exists
	    (void)Sys_TapeFile(unit, VTF_EjectTape, NULL);	// slightly recursive! can't fail
	    // actually create new tape
	    sysstate.vt[unit].tape = vtape_create(30);	// 30 minute tape
	    stat = (sysstate.vt[unit].tape == NULL);
	    if (!stat)
		Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    break;

	case VTF_LoadTape:
	    stat = Sys_TapeFile(unit, VTF_NewTape, NULL);	// slightly recursive!
	    if (stat == 0) {
		stat = vtape_fileread(sysstate.vt[unit].tape, filename);
		if (stat > 0)
		    UI_Alert("Error on line %d", stat);
	    }
	    if (stat == 0)
		Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    break;

	case VTF_SaveTape:
	    Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    stat = vtape_filewrite(sysstate.vt[unit].tape, filename);
	    break;

	case VTF_EjectTape:
	    // get rid of old tape if it exists
	    if (sysstate.vt[unit].tape)
		vtape_destroy(sysstate.vt[unit].tape);
	    sysstate.vt[unit].tape = NULL;
	    Sys_SetTapeProp(unit, TPROP_PLAYSTATE, EMPTY);
	    stat = 0;
	    break;

	default:
	    ASSERT(0);
    }

    Sys_GetTapeProp(unit, TPROP_POSITION, &pos);
    WinTapeNotify(unit, WT_TIME, pos);
    WinTapeNotify(unit, WT_FILESTATE, 0);
    return (stat != 0);
}


// heuristic test to see if a program is already running
int Sys_IsProgramRunning(void)
{
    return (simstate.prog_detect > 0);
}


// returns 1 if machine is in reset state
int Sys_MachineIsReset(void)
{
    return (simstate.runstate == RESET);
}


// null write -- used for personality module read-only memory

void
WrZ80_ROM(word Addr, byte Value)
{
    // do nothing
    ;
}


// a write to the display ram
void
WrZ80_display(word Addr, byte Value)
{
    int offset;

    ASSERT((Addr & 0xFC00) == 0xCC00);

    // part of script hack -- try to prevent keyboard polling
    // during output from eating up the script input
    // kind of heavyweight, but it only happens while a script
    // is running, so it doesn't matter all that much.
    if (simstate.kb_suppress) {
	TimerKill(simstate.kb_supptimer);
	simstate.kb_supptimer =
	    TimerCreate(TIMER_MS(SCRIPT_KB_SUPPTIMER), handle_kb_supp, 0, 0);
    }

    // see if value is unchanged -- eg, writing space on space
    if (Value == memimage[Addr])
	return;

    // update memory image
    memimage[Addr] = Value;

    // tell UI about it
    offset = Addr & 0x03FF;
    UI_UpdateDisplay(offset, Value);
}


// winmain calls this when keyboard input is received
// if we receive two notifications before the first one
// has been processed, the older one is lost.  the kb
// strobe nominally lasts 6 uS, so that if the key is read
// during that window, it will register more than once.
void
Sys_Keypress(byte data)
{
    if (simstate.kb_script) {
	// a script is running.  we ignore all input except ESC,
	// which will cancel the script.
	if (data == 0x1B) {
	    script_close(simstate.kb_sh);
	    simstate.kb_script =  0;
	    simstate.kb_sh     = -1;
	    if (simstate.kb_supptimer >= 0) {
		TimerKill(simstate.kb_supptimer);
		simstate.kb_suppress  =  0;
		simstate.kb_supptimer = -1;
	    }
	}
	return;
    }

    // normal processing
    sysstate.kb_stat  = 1;
    if (simstate.hkb_stat < 0)
	simstate.hkb_stat = TimerCreate(TIMER_US(6), handle_kb_stat, 0, 0);
    sysstate.kb_key = data;
}


// open up a script for keyboard input
// return 1 if OK, -1 on error.
int
Sys_Script(char *filename)
{
    int sh = script_open(filename,
			 SCRIPT_META_INC | SCRIPT_META_HEX | SCRIPT_META_KEY,
			 3);	// max nesting depth
    if (sh < 0)
	return -1;

    // flag indicating that a script is active now
    simstate.kb_script = 1;
    simstate.kb_sh     = sh;

    // cancel any pending key
    sysstate.kb_stat = 0;
    if (simstate.hkb_stat >= 0) {
	TimerKill(simstate.hkb_stat);
	simstate.hkb_stat = -1;	// dead timer
    }

    return 0;
}


// when the user interacts with the debugger, the major
// state changes come through here
void
Sys_DbgNotify(sysnotify_t event, int param)
{
    switch (event) {
	case DEBUG_KEY:
	    // halt no matter what and pop open debugger
	    simstate.runstate = HALTED;
	    simstate.inactivate_timer = 0;
	    UI_DbgWin(DW_Activate, 0);
	    report_curstate();
	    break;
	case RUN_KEY:
	    // run if we're halted, halt if we're running
	    if (simstate.runstate == RUNNING) {
		simstate.runstate = HALTED;
		simstate.inactivate_timer = 0;
		UI_DbgWin(DW_Activate, 0);
		report_curstate();
	    } else {
		simstate.runstate = RUNNING;
		startrun();	// establish normal/debugging fcn ptrs
		simstate.firststep = 1;
		simstate.inactivate_timer = 3;
	    }
	    break;
	case FORCERUN_KEY:
	    // run whether we are currently running or not
	    simstate.runstate = RUNNING;
	    startrun();	// establish normal/debugging fcn ptrs
	    simstate.firststep = 1;
	    simstate.inactivate_timer = 3;
	    break;
	case STEP_KEY:
	    // step over n instructions
	    if (simstate.runstate == RUNNING)
		UI_DbgWin(DW_Activate, 0);
	    simstate.runstate  = STEP;
	    simstate.stepcount = param;
	    simstate.firststep = 1;
	    startrun();	// establish normal/debugging fcn ptrs
	    simstate.inactivate_timer = 3;
	    break;
	case STEPI_KEY:
	    // step into n instructions
	    if (simstate.runstate == RUNNING)
		UI_DbgWin(DW_Activate, 0);
	    simstate.runstate  = STEPI;
	    simstate.stepcount = param;
	    simstate.firststep = 1;
	    simstate.inactivate_timer = 3;
	    startrun();	// establish normal/debugging fcn ptrs
	    break;
	case CLOSE_WIN:
	    simstate.runstate = RUNNING;
	    UI_DbgWin(DW_Close, 0);
	    break;
	default:
	    ASSERT(0);
    }
}


// -------------- fulfill emulator hooks ------------

// read a byte at the specified 16b address.
// any memory-mapped devices are visible.
byte
Sys_ReadMappedByte(word Addr)
{
    int page = (Addr >> 8);
    read_sub_t *rdhandler;

    ASSERT(Addr >= 0x0000);
    ASSERT(Addr <= 0xFFFF);

    rdhandler =	memmapRd[page];
    if (rdhandler == NULL)
	return memimage[Addr];
    else
	return (*rdhandler)(Addr);
}


// write a byte to the specified 16b address.
// any memory-mapped devices are visible.
void
Sys_WriteMappedByte(word Addr, byte Value)
{
    int page = (Addr >> 8);
    write_sub_t *wrhandler;

    ASSERT(Addr >= 0x0000);
    ASSERT(Addr <= 0xFFFF);

    wrhandler =	memmapWr[page];
    if (wrhandler == NULL)
	memimage[Addr] = Value;
    else
	(*wrhandler)(Addr, Value);
}


// like Sys_ReadMappedByte, except timing is considered
static byte
RdZ80_norm(word Addr)
{
    int page = (Addr >> 8);
    read_sub_t *rdhandler;

    ASSERT(Addr >= 0x0000);
    ASSERT(Addr <= 0xFFFF);

    // note: the SOL adds one wait state on any read in the range of C000-CFFF
    if ((Addr & 0xF000) == 0xC000)
	Z80Regs.ICount--;

    rdhandler =	memmapRd[page];
    if (rdhandler == NULL)
	return memimage[Addr];
    else
	return (*rdhandler)(Addr);
}


static void
WrZ80_norm(word Addr, byte Value)
{
    int page = (Addr >> 8);
    write_sub_t *wrhandler;

    ASSERT(Addr >= 0x0000);
    ASSERT(Addr <= 0xFFFF);

    // note: the SOL adds one wait state on any write in the range of C000-CFFF
    if ((Addr & 0xF000) == 0xC000)
	Z80Regs.ICount--;

    wrhandler =	memmapWr[page];
    if (wrhandler == NULL)
	memimage[Addr] = Value;
    else
	(*wrhandler)(Addr, Value);
}


// read a byte from whichever cassette is active, if either one is
// return one of the SVT_* codes.
// it has no sense of time; the caller is responsible for that.
static int
read_cassette(int unit)
{
    int stat;

    // it would be possible for both to be on, but unusual.
    // we just stick with vt[1] in that case.
    if (!(sysstate.vt[unit].motor_on || sysstate.vt[unit].motor_force) ||
	 (sysstate.vt[unit].playstate != PLAY))
	return SVT_NOBYTE;

    stat = vtape_readbyte(sysstate.vt[unit].tape,
			  sysstate.vt_baud,
			  &sysstate.vt_readbyte,
			  &sysstate.vt_overrun,
			  &sysstate.vt_framing);

    switch (stat) {
	case SVT_BYTE:
	    if (sysstate.vt_rxready)
		sysstate.vt_overrun = 1;	// we stacked up two
	    sysstate.vt_rxready = 1;
	    break;
	case SVT_NOBYTE:
	    break;
	case SVT_EOF:
	    Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    WinTapeNotify(unit, WT_EOF, 0);
	    break;
	default:
	    break;
    }

    return stat;
}


// whenver the cassette status port is read, we must see if we have
// to update the emulated cassette state.
static void
handle_vtape_status(void)
{
    int stat;	// FIXME: unused -- is this OK?

    if (!sysstate.vt_rxready && !simstate.vt_realtime) {
	int unit;
	for(unit=0; unit<2; unit++)
	    if ((sysstate.vt[unit].motor_on || sysstate.vt[unit].motor_force) &&
		(sysstate.vt[unit].playstate == PLAY)) {
		stat = read_cassette(unit);
		break;
	    }
    }
}


// some programs (like the Music System) misbehave if back to back
// keyboard status polls present a new character.  If we are scripting,
// we have N dead cycles between each key returned.
#define SCRIPT_KEY_DUTYPERIOD (10)

static void
handle_scripting(void)
{
    static int duty_cycle = 0;
    char ch;
    int rc;

    duty_cycle++;
    if (duty_cycle >= SCRIPT_KEY_DUTYPERIOD)
	duty_cycle = 0;

    if (!simstate.kb_script   ||	// not running a script
	 sysstate.kb_stat     ||	// already have a keystroke pending
	 simstate.kb_suppress ||	// a short time after a <CR>
	(duty_cycle > 0)) 		// don't return a key every cycle
	return;

    rc = script_next_char(simstate.kb_sh, &ch);
    if (rc != SCRIPT_OK) {
	script_close(simstate.kb_sh);
	simstate.kb_script = 0;
	simstate.kb_sh     = -1;
	if (simstate.kb_supptimer >= 0) {
	    TimerKill(simstate.kb_supptimer);
	    simstate.kb_suppress  =  0;
	    simstate.kb_supptimer = -1;
	}
	return;
    }

    sysstate.kb_stat = 1;
    sysstate.kb_key  = ch;

    // Heuristic:
    //
    // SOLOS, and other programs, sometimes poll the keyboard during
    // printing/scrolling to see if the user has hit the space bar or
    // the MODE key or somesuch.  The problem is that this polling also
    // eats the script input, which probably isn't intended.
    //
    // The heuristic is to note when we stuff a <CR>, which is likely to
    // be what triggers lots of output.  We just set a timer to suppress
    // getting input from the script for a little while.  If we note any
    // screen output while this timer is active, we reset the timer.
    // Eventually the output should stop and the timer times out and we
    // start stuffing keystrokes again.
    //
    // Not perfect, but much better than doing nothing.
    ASSERT(simstate.kb_supptimer < 0);
    if (ch == 0x0D) {
	simstate.kb_supptimer =
	    TimerCreate(TIMER_MS(SCRIPT_KB_SUPPTIMER), handle_kb_supp, 0, 0);
	simstate.kb_suppress = 1;
    }
}


static byte
InZ80_norm(word Port)
{
    read_sub_t *rdhandler;

    ASSERT(Port >= 0x00);
    ASSERT(Port <= 0xFF);

    // note: the SOL adds one wait state on any IN operation
    Z80Regs.ICount--;

    rdhandler =	memmapIn[Port];
    if (rdhandler != NULL) {
	return (*rdhandler)(Port);
    } else {
	// ignored
#ifdef DEBUG
	// FIXME: pop up warning window, at least the first time?
	printf("Warning: read from unimplemented IO port 0x%02x\n", Port);
#endif
	return 0xFF;	// if nothing drives it, the bus floats to ones
    }
}


static void
OutZ80_norm(word Port, byte Value)
{
    write_sub_t *wrhandler;

    // note: the SOL adds one wait state on any OUT operation
    Z80Regs.ICount--;

    ASSERT(Port >= 0x00);
    ASSERT(Port <= 0xFF);

    wrhandler =	memmapOut[Port];
    if (wrhandler != NULL) {
	(*wrhandler)(Port, Value);
    } else {
	// ignored
#ifdef DEBUG
	// FIXME: pop up warning window, at least the first time?
	printf("Warning: write to unimplemented IO port 0x%02x\n", Port);
#endif
    }
}

// ========================================================================
// system modelling
// ========================================================================

// 250 ms after scrolling, turn off scroll timer bit
static void
handle_outfe_timer(uint32 arg1, uint32 arg2)
{
    sysstate.scrolltimer = 0;
}


// suppress reading script input while program is outputting
static void
handle_kb_supp(uint32 arg1, uint32 arg2)
{
    simstate.kb_suppress  =  0;
    simstate.kb_supptimer = -1;
}


// simulate a 10 uS or so strobe after keypress
static void
handle_kb_stat(uint32 arg1, uint32 arg2)
{
    simstate.hkb_stat = -1;	// inactive
}


// cassette uart rx channel timer callback.
// this function handles FWD and RWD states too.
// arg1 is the tape unit
static void
handle_svt_rdtimer(uint32 arg1, uint32 arg2)
{
    // one char at  300 baud is 33.333 ms
    // one char at 1200 baud is  8.333 ms
    int time, stat, pos, realtime;
    int unit = arg1;

    simstate.vt_rdtimer[unit] = -1;	// inactivate by default
    if (!(sysstate.vt[unit].motor_on || sysstate.vt[unit].motor_force))
	return;

    Sys_GetTapeProp(unit, TPROP_POSITION, &pos);
    Sys_GetTapeProp(unit, TPROP_REALTIME, &realtime);

    switch (sysstate.vt[unit].playstate) {

	case EMPTY:
	case STOP:
	case RECORD:
	    // do nothing except disable timer
	    break;

	case PLAY:
	    if (realtime)
		stat = read_cassette(unit);
	    else
		stat = (sysstate.vt_rxready) ? SVT_OK : read_cassette(unit);
	    if (stat != SVT_EOF) {
		// even if we are !realtime, we schedule a callback
		// to make sure that time advances at least as fast
		// as realtime.  this is important because SOLOS goes
		// into a ~1 second idle loop after the cassette motor
		// is first turned on.  we don't want the timer to freeze
		// in this mode because we aren't polling the status reg.
		time = (sysstate.vt_baud == 1200) ? 8333 : 33333;
		simstate.vt_rdtimer[unit] =
		    TimerCreate(TIMER_US(time), handle_svt_rdtimer, unit, arg2);
	    }
	    Sys_GetTapeProp(unit, TPROP_POSITION, &pos);
	    WinTapeNotify(unit, WT_TIME, pos);
	    break;

	case FWD:
	    // every 10th of a second, adjust pointer.  if not realtime, go 10x faster.
	    pos++;
	    if (vtape_setprop(sysstate.vt[unit].tape, VTPROP_POSITION, pos) == SVT_EOF) {
		// hit end of tape
		Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
		WinTapeNotify(unit, WT_EOF, 0);
	    } else {
		time = (realtime) ? 10000 : 1000;
		simstate.vt_rdtimer[unit] =
		    TimerCreate(TIMER_US(time), handle_svt_rdtimer, unit, arg2);
		WinTapeNotify(unit, WT_TIME, pos);
	    }
	    break;

	case REW:
	    // every 10th of a second, adjust pointer
	    pos--;
	    if (pos <= 0) {
		// hit start of tape
		pos = 0;
		Sys_SetTapeProp(unit, TPROP_POSITION, pos);
		Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    } else {
		time = (realtime) ? 10000 : 1000;
		Sys_SetTapeProp(unit, TPROP_POSITION, pos);
		simstate.vt_rdtimer[unit] =
		    TimerCreate(TIMER_US(time), handle_svt_rdtimer, unit, arg2);
	    }
	    WinTapeNotify(unit, WT_TIME, pos);
	    break;

	default:
	    ASSERT(0);

    } // switch
}


// cassette uart tx channel timer callback.
//
// if the tape TX UART channel has data ready, write that, otherwise
// just record carrier tone to the tape at the current baud rate.
//
// it is theoretically possible that both tape players are in the
// RECORD state and that both have their motor_on set (although
// SOLOS doesn't do this).  in this case, only one of the units
// actually gets written to as the first one whose timer triggers
// will record the byte.
static void
handle_svt_wrtimer(uint32 arg1, uint32 arg2)
{
    // one char at  300 baud is 33.333 ms
    // one char at 1200 baud is  8.333 ms
    int time = (sysstate.vt_baud == 1200) ? 8333 : 33333;
    int stat, pos;
    int data;		// byte value to save
    int carrierflag;	// -1 if carrier instead of data
    int unit = arg1;
    int recording = 0;	// default

    simstate.vt_wrtimer[unit] = -1;	// inactivate by default

    if ((sysstate.vt[unit].motor_on || sysstate.vt[unit].motor_force) &&
	(sysstate.vt[unit].playstate == RECORD)) {

	if (sysstate.vt_TTBE) {
	    // transmit buffer empty
	    data = 0x00;
	    carrierflag = -1;
	} else {
	    data = sysstate.vt_writebyte;
	    carrierflag = SVT_OK;
	}

	stat = vtape_writebyte(sysstate.vt[unit].tape,
			       sysstate.vt_baud,
			       data, carrierflag);

	if (stat == SVT_EOF) {
	    Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	    WinTapeNotify(unit, WT_EOF, 0);
	} else {
	    Sys_GetTapeProp(unit, TPROP_POSITION, &pos);
	    WinTapeNotify(unit, WT_TIME, pos);
	    recording = 1;
	}

    }

    sysstate.vt_TTBE = 1;	// UART TX buffer is now empty
    // this is where we aren't true to life in the exceptional
    // case where both cassette's motors are on and recording.

    // schedule next deadline to record carrier.
    // this timer might be killed later if !realtime and a new
    // byte comes in to be written.
    if (recording)
	simstate.vt_wrtimer[unit] =
	    TimerCreate(TIMER_US(time), handle_svt_wrtimer, unit, 0);
}


/*
   Bit 1 of port 0xFE contains the active high signal "scan adv"
   (scanline advance).  It is related to, but not the same as,
   horizontal blanking.  The waveforms look like this, with the
   numbers below specifying time in characters:


      vid       ________--------------------________________

                   13            64                25


      scan adv  ____________________________----------------_

                            13+64                  25

   each char time is 14.31 MHz * 9 dots, or 628.93 ns.
*/
static void
handle_scanadv(uint32 arg1, uint32 arg2)
{
    int ticks;
    sysstate.scanadv = !sysstate.scanadv;
    ticks = (sysstate.scanadv) ? 25*9 : 77*9;
    simstate.hscanadv = TimerCreate(ticks, handle_scanadv, 0, 0);
}


// read I/O device 0xF8: read serial port status
static byte
handle_IO_In_F8(word Addr)
{
    // bit 0: serial carrier detect (active low)
    // bit 1: serial data set ready (active low)
    // bit 2: PE: Parity Error--received parity does not compare to
    //            that programmed.
    // bit 3: FE: Framing Error--valid stop bit not received when
    //            expected.
    // bit 4: OE: Overrun Error--CPU did not accept data before it
    //            was replaced with additional data.
    // bit 5: serial clear to send (active low)
    // bit 6: DR: Data Ready--data received by UART is available
    //            when requested.
    // bit 7: TBRE: Transmitter Buffer Register Empty--UART is ready
    //            to accept another word from the Bidirectional
    //            Data Bus.
    // FIXME
    return   (1<<7)	//  TBRE -- for now, swallow everything
	   | (0<<6)	// !DR   -- for now, no input
	   | (0<<5)	//  CTS  (active low)
	   | (0<<4)	// !OE
	   | (0<<3)	// !FE
	   | (0<<2)	// !PE
	   | (0<<1)	//  DSR  (active low)
	   | (0<<0)	//  CD   (active low)
	;
}

// read I/O device 0xF9: read byte from serial port
static byte
handle_IO_In_F9(word Addr)
{
    return 0x00;	// FIXME
}

// read I/O device 0xFA: read parallel port, keyboard, and cassette status
static byte
handle_IO_In_FA(word Addr)
{
    handle_vtape_status();
    handle_scripting();
    return (sysstate.vt_TTBE        << 7)	// cassette TTBE (tape tx buffer empty)
	 | (sysstate.vt_rxready     << 6)	// cassette TDR  (tape data ready)
	 | (0                       << 5)	// unused
	 | (sysstate.vt_overrun     << 4)	// cassette OE   (read overrun error)
	 | (sysstate.vt_framing     << 3)	// cassette FE   (read framing error)
	 | (0                       << 2)	// parallel port device ready (active low)
	 | (1                       << 1)	// parallel port data input ready (active low)
	 | ((~sysstate.kb_stat & 1) << 0)	// KDR (keyboard data ready)
	;
}

// read I/O device 0xFB: read byte from cassette uart
static byte
handle_IO_In_FB(word Addr)
{
    sysstate.vt_rxready = 0;
    sysstate.vt_overrun = 0;
    sysstate.vt_framing = 0;
    return sysstate.vt_readbyte;
}

// read I/O device 0xFC: read keyboard
static byte
handle_IO_In_FC(word Addr)
{
    if (simstate.hkb_stat < 0)
	sysstate.kb_stat = 0;
    return sysstate.kb_key;
}

// read I/O device 0xFD: read byte from parallel port and clear !RDY flop
static byte
handle_IO_In_FD(word Addr)
{
    return 0x00;	// FIXME
}

// read I/O device 0xFE: return video display scroll timer
static byte
handle_IO_In_FE(word Addr)
{
    // bit 0: 250 ms timer after each OUT FE (active high)
    // bit 1: horizontal sync signal (active high)
    //        blank 13 chars, video 64 chars, blank & sync 25 chars
    //        == 8.17 us,     == 40.23 us,    == 15.71 us
    //        so active 15.71 us/64.11 us, or 24.5%
    // because hscanadv makes so many transitions, it is scheduled
    // only when it becomes necessary.
    if (simstate.hscanadv < 0)
	simstate.hscanadv = TimerCreate(100, handle_scanadv, 0, 0);	// start it first time
    return (sysstate.scanadv << 1)
	|  (sysstate.scrolltimer);
}

// read I/O device 0xFF: read sense switches -- active low
static byte
handle_IO_In_FF(word Addr)
{
    return (~Sys_GetDipswitch(2)) & 0xFF;
}


// write I/O device 0xF8: set serial port RTS
static void
handle_IO_Out_F8(word Addr, byte Value)
{
    // bit 4: serial port RTS (ready to send) (active high)
    // FIXME
}

// write I/O device 0xF9: write byte to serial port
static void
handle_IO_Out_F9(word Addr, byte Value)
{
    // FIXME
}

// write I/O device 0xFA: write parallel port and cassette data control bits
static void
handle_IO_Out_FA(word Addr, byte Value)
{
    // bit 0: (unused)
    // bit 1: (unused)
    // bit 2: (unused)
    // bit 3: PIE (Parallel Input Enable, active low, resets to 1)
    // bit 4: PUS (Parallel Unit Select, active high, resets to 0)
    // bit 5: tape speed (0=1200 baud, 1=300 baud, resets to 0)
    // bit 6: tape 2 motor enable (active high acc'd to SOLOS, low acc'd to schematic, resets to 0)
    // bit 7: tape 1 motor enable (active high acc'd to SOLOS, low acc'd to schematic, resets to 0)
    // FIXME
    sysstate.vt_baud        = (Value & 0x20) ? 300 : 1200;
    sysstate.vt[0].motor_on = (Value >> 6) & 1;
    sysstate.vt[1].motor_on = (Value >> 7) & 1;
    if (sysstate.vt[0].motor_on && sysstate.vt[1].motor_force) {
	// FIXME: don't just warn about it
	static flag = 1;
	if (flag) {
	    flag = 0;
	    UI_Alert("Warning: both cassette motors enabled -- emulator might not model this properly");
	}
    }
}

// write I/O device 0xFB: write byte to cassette uart
static void
handle_IO_Out_FB(word Addr, byte Value)
{
    int unit;

    // see if the uart is still busy writing the previous byte
    if (!sysstate.vt_TTBE) {
	static flag = 1;
	if (flag) {
	    UI_Alert("Warning: program at $%04X wrote to cassette data out port while it was busy", Z80Regs.PC.W);
	    flag = 0;
	}
    }

    // write it to tape
    sysstate.vt_TTBE      = 0;	// TX busy
    sysstate.vt_writebyte = Value;

    for(unit=0; unit<2; unit++) {
	if ((sysstate.vt[unit].motor_on || sysstate.vt[unit].motor_force) &&
	    (sysstate.vt[unit].playstate == RECORD)) {

	    if (simstate.vt_realtime) {
		if (simstate.vt_wrtimer[unit] < 0) {
		    int time = (sysstate.vt_baud == 1200) ? 8333 : 33333;
		    simstate.vt_wrtimer[unit] =
			TimerCreate(TIMER_US(time), handle_svt_wrtimer, unit, 0);
		    break;	// schedule only one -- two recording at once won't work anyway
		}

	    } else {
		// if we're waiting to record carrier, kill it
		if (simstate.vt_wrtimer[unit] >= 0)
		    TimerKill(simstate.vt_wrtimer[unit]);
		handle_svt_wrtimer(unit, 0);	// write it directly
	    }

	} // if (this unit is recording)
    } // for(unit)
}

// write I/O device 0xFD: write byte to parallel port
static void
handle_IO_Out_FD(word Addr, byte Value)
{
    // FIXME
}

// write I/O device 0xFE: set display scrollbase
static void
handle_IO_Out_FE(word Addr, byte Value)
{
    if (sysstate.scrolltimer == 1)
	TimerKill(simstate.hscrolltimer);	// one is already active
    sysstate.scrolltimer = 1;
    simstate.hscrolltimer =
	    TimerCreate(TIMER_MS(250), handle_outfe_timer, 0, 0);

    sysstate.scrollbase  = ((Value >> 0) & 0xF);
    sysstate.scrollshade = ((Value >> 4) & 0xF);
    UI_UpdateScrollBase(sysstate.scrollshade, sysstate.scrollbase);
}

// write I/O device 0xFF: optional
static void
handle_IO_Out_FF(word Addr, byte Value)
{
    // dudley henderson had a tricked-up Sol for editing
    // tibetan text.  He modified the character generation
    // logic to choose between one of two fonts, toggling
    // between them every time OUT 0xFF was performed.
    //
    // this is a bit impure, but we let the winmain code
    // handle all the display generation logic.
    if (UI_OutFFTickle())
	return;

// FIXME: instead of doing this, enable/disable this port
//        based on user setting/clearing the config bit
#ifdef DEBUG
    printf("Warning: write to unimplemented IO port 0x%02x\n", Port);
#endif
}

// ========================================================================
//  generic emulation details
// ========================================================================

static word
Exec8080_brkpt(Z80 *R)
{
    int addr = R->PC.W;
    word retval;

    if (BRKPT_ADDR(addr) && !simstate.firststep) {

	// potentially a hit -- do slow check
	int hit = breakpoint_test(BRK_OP, R->PC.W, (word)0, &R->brknum);

	if (hit) {
	    R->brkpt = 1;
	    R->reason = BRKPT_BRK;
	    simstate.firststep = 0;
	    return R->PC.W;
	}

    } // if (BRKPT_ADDR())

    // simulate the instruction
    retval = Exec8080(R);

    if (R->brkpt && R->reason == BRKPT_BRK) {
	// validate breakpoint (rd16 and wr16 are troublesome)
	int hit = breakpoint_verify(&R->brknum);
	if (!hit)
	    R->brkpt = FALSE;	// oops, false alarm
    }

    simstate.firststep = 0;	// clear it in case this is the first time
    return retval;
}


#if 0
// This is not used because the PC breakpoints are detected before
// calling the Exec8080() routine, where this would be called.
// This is done because otherwise we'd have to have breakpoint
// checking in Exec8080() which would consume time even if we
// weren't single stepping.
static byte
OpZ80_brkpt(word Addr)
{
    return RdZ80_norm(Addr);	// FIXME
}
#endif


static byte
RdZ80_brkpt(word Addr)
{
    byte data = RdZ80_norm(Addr);

    if (BRKPT_ADDR(Addr)) {

	// potentially a hit -- do slow check
	int hit = breakpoint_test(BRK_RD, Addr, (word)data, &Z80Regs.brknum);

	if (hit) {
	    Z80Regs.brkpt = 1;
	    Z80Regs.reason = BRKPT_BRK;
	}
    }

    return data;
}


static void
WrZ80_brkpt(word Addr, byte Value)
{
    WrZ80_norm(Addr, Value);

    if (BRKPT_ADDR(Addr)) {

	// potentially a hit -- do slow check
	int hit = breakpoint_test(BRK_WR, Addr, (word)Value, &Z80Regs.brknum);

	if (hit) {
	    Z80Regs.brkpt = 1;
	    Z80Regs.reason = BRKPT_BRK;
	}
    }
}


static byte
InZ80_brkpt(word Port)
{
    byte data = InZ80_norm(Port);

    if (BRKPT_IO(Port)) {

	// potentially a hit -- do slow check
	int hit = breakpoint_test(BRK_IN, Port, (word)data, &Z80Regs.brknum);

	if (hit) {
	    Z80Regs.brkpt = 1;
	    Z80Regs.reason = BRKPT_BRK;
	}
    }

    return data;
}


static void
OutZ80_brkpt(word Port, byte Value)
{
    OutZ80_norm(Port, Value);

    if (BRKPT_IO(Port)) {

	// potentially a hit -- do slow check
	int hit = breakpoint_test(BRK_OUT, Port, (word)Value, &Z80Regs.brknum);

	if (hit) {
	    Z80Regs.brkpt = 1;
	    Z80Regs.reason = BRKPT_BRK;
	}
    }
}


// ---------------------------------------
//   simulation control
// ---------------------------------------


// execute one instruction and return
void
simstep(void)
{
    int t1;

    Z80Regs.startPC = Z80Regs.PC;

    t1 = Z80Regs.ICount;
    (void)(simstate.exec8080_func)(&Z80Regs);

    TimerTick(t1 - Z80Regs.ICount);
}


// run for Z80Regs.ICount simulated 8080 clock cycles and return
void
simrun(void)
{
    int t1;

    while (Z80Regs.ICount > 0 && !Z80Regs.brkpt) {
	Z80Regs.startPC = Z80Regs.PC;
	t1 = Z80Regs.ICount;
	(void)(simstate.exec8080_func)(&Z80Regs);
	TimerTick(t1 - Z80Regs.ICount);
    }
}


// step over simstate.stepcount instructions or until ICount
// simulated 8080 cycles have elapsed.
void
simstepover(void)
{
    int t1;
    byte nextop;

    while (Z80Regs.ICount > 0 && !Z80Regs.brkpt) {

	Z80Regs.startPC = Z80Regs.PC;
	t1 = Z80Regs.ICount;

	if (simstate.runstate == STEP) {
	    nextop = DiZ80(Z80Regs.PC.W);
	    if (  (nextop         == 0xCD) || // call
		 ((nextop & 0xC7) == 0xC4)) { // call conditional
		breakpoint_temp_add((word)(Z80Regs.PC.W + 3));  // just past call
		simstate.runstate = STEPRUN;	// run until breakpoint
	    }
	}

	(void)(simstate.exec8080_func)(&Z80Regs);
	TimerTick(t1 - Z80Regs.ICount);

	if (Z80Regs.brkpt) {

	    if (Z80Regs.brknum == 0) {
		// ok, we're not really done, we've just hit the
		// step-over temporary breakpoint.  note we didn't
		// perform the operation, so don't dec stepcount.
		breakpoint_temp_kill();
		simstate.runstate = STEP;
		simstate.stepcount--;
	    } else {
		// we hit a real breakpoint
		return;
	    }

	} else if (simstate.runstate == STEP)
	     simstate.stepcount--;

	if (simstate.stepcount < 1) {
	    // we've stepped the appropriate number of times;
	    // go back to debugger mode
	    Z80Regs.ICount = 0;
	    breakpoint_temp_kill();
	    UI_DbgWin(DW_ActivateStep, 0);
	    simstate.runstate = HALTED;
	    report_curstate();
	    return;
	}

    } // while

    // we ran out of cycles in this time slice, but leave active
    // the STEP state and the remaining number of stepcounts.
}


// report why we've stopped
static void
brkpt_notify(void)
{
    char buf[80];

    // if it was a temporary breakpoint (run-to address, single step),
    // don't squawk about it
    if (Z80Regs.brknum == 0)
	return;

    UI_DbgWin(DW_Activate, 0);

    switch (Z80Regs.reason) {
	case BRKPT_BRK:
	    sprintf(buf, "  Processor hit breakpoint %d at PC=$%04X.", Z80Regs.brknum, Z80Regs.startPC.W);
	    UI_DbgWinLog(buf, FALSE);
	    sprintf(buf, "blist %d", Z80Regs.brknum);
	    dbg_interp(buf);
	    break;
	case BRKPT_ILLEGAL:
	    sprintf(buf, "  Processor executed undefined instruction at PC=$%04X.\n  You must reset the Sol.", Z80Regs.PC.W);
	    UI_DbgWinLog(buf, FALSE);
	    break;
	case BRKPT_HALT:
	    sprintf(buf, "  Processor executed HALT instruction at PC=$%04X.\n  You must reset the Sol.", Z80Regs.PC.W);
	    UI_DbgWinLog(buf, FALSE);
	    break;
	case BRKPT_DEBUG:
	    sprintf(buf, "  Processor stopped for debugging at PC=$%04X.\n  You must reset the Sol.", Z80Regs.PC.W);
	    UI_DbgWinLog(buf, FALSE);
	    break;
	default:
	    break;
    }
}


// this is called every time we transition to the halted state
static void
report_curstate(void)
{
    char buf[120];
    int len;

    len = sprintf(buf, "  A=%02X F=%c%c%c%c%c%c BC=%04X DE=%04X HL=%04X SP=%04X PC=%04X ",
		      Z80Regs.AF.B.h,
		    ((Z80Regs.AF.B.l) & 0x80) ? 'S' : '-',
		    ((Z80Regs.AF.B.l) & 0x40) ? 'Z' : '-',
		    ((Z80Regs.AF.B.l) & 0x10) ? 'H' : '-',
		    ((Z80Regs.AF.B.l) & 0x04) ? 'P' : '-',
		    ((Z80Regs.AF.B.l) & 0x02) ? 'N' : '-',
		    ((Z80Regs.AF.B.l) & 0x01) ? 'C' : '-',
		      Z80Regs.BC.W,
		      Z80Regs.DE.W,
		      Z80Regs.HL.W,
		      Z80Regs.SP.W,
		      Z80Regs.PC.W
		);

#if 1
    // add disassembly to tail end of buf
    (void)DAsm(&buf[len], (word)Z80Regs.PC.W, 0);
#endif

    UI_DbgWinLog(buf, FALSE);
}


static void
KickCassettes(void)
{
    // kick off cassette timers if needed.  it is possible
    // that both tape players are ON and playing, but this
    // really isn't expected.
    int unit;
    for(unit=0; unit<2; unit++) {

	if (!sysstate.vt[unit].motor_on && !sysstate.vt[unit].motor_force)
	    continue;

	// kick off tape RX channel if it isn't going
	if ((simstate.vt_rdtimer[unit] < 0) &&
	    (sysstate.vt[unit].playstate != EMPTY) &&
	    (sysstate.vt[unit].playstate != RECORD) &&
	    (sysstate.vt[unit].playstate != STOP))
		handle_svt_rdtimer(unit, 0);

	// kick off tape TX channel if it isn't going
	if ((simstate.vt_wrtimer[unit] < 0) &&
	    (sysstate.vt[unit].playstate == RECORD))
		handle_svt_wrtimer(unit, 0);

    } // for(unit)
}


// run the emulator core until the user wants to quit.
//
// this continually runs a timeslice's worth of 8080 cycles, then it
// resynchronizes to the real-world timeslice duration.  it is during
// during this resynchronization that windows messages get handled.
// MAX_CATCHUP defines how many time slices we will attempt to make
// up for, but if we ever fall further behind than that, we just
// live with the fact that we blew it.  This prevents us from running
// virtually unregulated for a long time after some serious speed hit
// (eg, lots of scrolling, or the user does a menu selection).
#define MAX_CATCHUP 4

void
Sys_DoTimeSlicing(void)
{
    int32 slice_start;		// time, in real ms, of when slice began
    int   sol_timeslip = 0;	// accumulated time difference, in 8080 ticks
    int   x86_timeslip = 0;	// accumulated time difference, in ms
    int   x86_ms_tot = 0;	// # of ms since last report of x86 CPU time spent working
    int   real_ms_tot = 0;	// # of ms of realtime since last report
    int   sol_ticks_tot = 0;	// # of 8080 clock ticks since last report
    int   done = 0;

    simstate.tick_freq = HostQueryTimebase();	// get system timer frequency

    simstate.timeslice_ms = 10;			// in milliseconds

    if (simstate.tick_freq == 0) {
	UI_Alert("This machine doesn't support a high-precision timer.\n" \
		 "Running without speed regulation.\n");
	simstate.up_speed = MHZ_unregulated;
    }
    UpdateTimerInfo();

    slice_start = HostGetRealTimeMS();

    startrun();

    while (!done) {

	int32 slice_end;	// time, in real ms, of when slice ended
	int32 work_end;		// time, in real ms, of when emulation work ended
	int i;


	if (Z80Regs.brkpt) {
	    Z80Regs.brkpt = 0;		// don't trip on it again
	    brkpt_notify();
	    breakpoint_temp_kill();	// in case it came from temp breakpoint
	    UI_DbgWin(DW_Activate, 0);
	    simstate.runstate = HALTED;
	    report_curstate();
	}

	if (simstate.runstate != RUNNING) {
	    // this message drain is required if we are in the debugger
	    // stopped or single stepping.  if we are RUNNING, the message
	    // queue gets drained during the time we are idling.
	    while (!done && UI_HandleMessages(&done))
		;
	    // reset statistics counters
	    x86_ms_tot = 0;
	    real_ms_tot = 0;
	    sol_ticks_tot = 0;
	}

	if (simstate.runstate == HALTED)
	    continue;

	if (simstate.runstate == RESET) {
	    if (!UI_KBResetState())
		simstate.runstate = RUNNING;
	    continue;
	}

	if (simstate.runstate == STEPI) {
	    int i;
	    simstate.firststep = 1;
	    for(i=0; i<simstate.stepcount; i++) {
		simstep();
		if (Z80Regs.brkpt)
		    break;
	    }
	    UI_DbgWin(DW_ActivateStep, 0);
	    simstate.runstate = HALTED;
	    report_curstate();
	    continue;
	}

	ASSERT( (simstate.runstate == RUNNING) ||
		(simstate.runstate == STEPRUN) ||
		(simstate.runstate == STEP) );

	// simulate for one timeslice's worth of 8080 instructions
	if (simstate.tick_freq != 0) {

	    KickCassettes();	// kick off cassette operation if appropriate

	    // # of 8080 clocks to perform, accounting for a few extra
	    // ticks we may have performed in the previous slice
	    Z80Regs.ICount = (uint32)(simstate.sol_ticks_per_slice + sol_timeslip);

	    // if we are rate limited, count the target 8080 cycle count,
	    // independent of any fudging we need to do to keep the audio
	    // sample buffers in range.
	    if (simstate.up_speed != MHZ_unregulated)
		sol_ticks_tot += Z80Regs.ICount;

	    // slow down the 8080 emulation to keep sync with audio
	    switch (simstate.throttle) {
		case Throttle_NORMAL: break;
		case Throttle_FASTER: Z80Regs.ICount += (Z80Regs.ICount*8)/16; break;
		case Throttle_SLOWER: Z80Regs.ICount -= (Z80Regs.ICount*8)/16; break;
		default: ASSERT(0); break;
	    }
	    // if we aren't throttled, count the real 8080 cycle count
	    if (simstate.up_speed == MHZ_unregulated)
		sol_ticks_tot += Z80Regs.ICount;

	    // run for R->ICount 8080 cycles
	    if (simstate.runstate == RUNNING)
		simrun();
	    else
		simstepover();
	    sol_timeslip = Z80Regs.ICount;	// may be negative

	    // detect if PC is in personality module space or not.
	    // this info is used heuristically to know if we should
	    // allow the user to load a program.
	    if ((Z80Regs.PC.W < 0xC000) || (Z80Regs.PC.W > 0xC7FF))
		simstate.prog_detect = 30;	// reset timer
	    else if (simstate.prog_detect > 0)
		simstate.prog_detect--;

	    intaudio_advance_time();

	    // drain any messages that we have triggered/generated
	    while (!done && UI_HandleMessages(&done))
		;

	    // note when we are done working, before killing time
	    slice_end = work_end = HostGetRealTimeMS();

	    // if we are running ahead of schedule, kill time time
	    if (simstate.up_speed != MHZ_unregulated) {
		int idle_ms = (slice_start + simstate.timeslice_ms)	// target end time
			    -  slice_end				// actual end time
			    -  x86_timeslip;
		if (idle_ms < 0)
		    idle_ms = 0;
		Sleep(idle_ms);
		// Sleep() promises not to undershoot, but can overshoot terribly.
		// figure out where we stand again.
		slice_end = HostGetRealTimeMS();
	    }

	    // figure out how much our realtime goal is ahead or behind
	    real_ms_tot  += (slice_end - slice_start);	// realtime
	    x86_ms_tot   += (work_end  - slice_start);	// CPU work time
	    x86_timeslip += (slice_end - slice_start - simstate.timeslice_ms);	// simulation time error

	    // don't let the correction get too far out of whack
	    if (x86_timeslip > MAX_CATCHUP*simstate.timeslice_ms)
		x86_timeslip = MAX_CATCHUP*simstate.timeslice_ms;
	    else if (x86_timeslip < -MAX_CATCHUP*simstate.timeslice_ms)
		x86_timeslip = -MAX_CATCHUP*simstate.timeslice_ms;

	    // every realtime second or so update the status bar with some
	    // emulation speed statistics
	    if (real_ms_tot >= 1000) {
		char buf[80];
		sprintf(buf, "Simulated speed: %6.2f MHz",
			(sol_ticks_tot * 1E-6) / (real_ms_tot * 1E-3));
		if (simstate.up_speed != MHZ_unregulated) {
		    int load = (int)((100.0f*(float)x86_ms_tot) / (float)real_ms_tot);
		    sprintf(buf+strlen(buf),"      x86 load: %3d%%", load);
		}
		UI_NoteSpeed(buf);

		// restart statistics counters
		sol_ticks_tot = 0;
		x86_ms_tot = 0;
		real_ms_tot = 0;
	    }

	} else {

	    // we don't have an accurate timer to work with, so just run
	    // unregulated -- simulate 1000 instructions ASAP
	    for(i=0; i<1000 && (!Z80Regs.brkpt); i++)
		simstep();

	    intaudio_advance_time();

	    // now drain any messages
	    while (!done && UI_HandleMessages(&done))
		;
	}

	// we could have done this before we started running, but often
	// times (esp single step, or running with breakpoints) we don't
	// go through even one time slice before we halt again, causing
	// excessive flashing.  instead, we wait a little bit before
	// disabling the debug window.
	if (simstate.inactivate_timer > 0) {
	    simstate.inactivate_timer--;
	    if (simstate.inactivate_timer == 0)
		UI_DbgWin(DW_Inactivate, 0);
	}

	slice_start = slice_end;

#if 0
	// report what the actual loop time was
	{
	    char buf[100];
	    int true_end = HostGetRealTimeMS();
	    sprintf(buf, "%d ms", true_end - slice_start);
	    UI_NoteSpeed(buf);
	}
#endif
    } // while(!done)
}


// compute timeslice info after something has changed
void
UpdateTimerInfo(void)
{
    // the Sol timebase is based on a 14.31818 MHz colorburst crystal
    const double colorburst = 14318180.0;

    switch (simstate.up_speed) {
	case MHZ_unregulated:
	    // in some ways, unregulated should be independent of the
	    // emulated 8080 frequency.  for instance, the vdisk_ns stuff
	    // has some idea of real time which is in terms of emulated
	    // 8080 clock cycles, not the wallclock time.  changing the
	    // emulated 8080 frequency will mean more or fewer emulated
	    // 8080 cycles pass in an emulated second, but that is entirely
	    // independent of wall-clock time.
	    //
	    // so, rather than getting into all that, I'll just pick it
	    // to map to the most common CPU frequency.
	    // ... fall through ...
	case MHZ_2_04:
	    simstate.sol_ticks_per_ms = (int32)((colorburst/7.0) / 1000.0);
	    TimerTickScale(7);
	    break;
	case MHZ_2_38:
	    simstate.sol_ticks_per_ms = (int32)((colorburst/6.0) / 1000.0);
	    TimerTickScale(6);
	    break;
	case MHZ_2_86:
	    simstate.sol_ticks_per_ms = (int32)((colorburst/5.0) / 1000.0);
	    TimerTickScale(5);
	    break;
	default:
	    ASSERT(0);
    }
    simstate.sol_ticks_per_slice = simstate.timeslice_ms * simstate.sol_ticks_per_ms;
}
